function [base]    =   RABT_Base(scrsize,folder)
% RABT_Base
% 
% Author: Koen Hilgersom (2011)
% 01/07/2011
%
% Function file is run by the script 'Rising Air Bubble Technique_vX_X.m'
%
% This function file creates or loads a project (based on the folder path inserted
% by the user) and sets up or loads the base grid for the transformation of the 
% pictures within this project.
% One project has a constant rising velocity, amount of nozzles and base grid 
% (in case these parameters were changed during the measurement, please load these 
% pictures to another project folder).
%
% Input
% scrsize    -    the user's screen size as determined by Matlab in the
%                   main script
% folder     -    path name of the folder in which the images are loaded
%
% Output
% base       -    variables for the specific situation
% RABT       -    variables for the photo
%
%---------------------------------------------------------------------------  
%      Copyright (C) 2011 Technische Universiteit Delft, 
%          Koen Hilgersom
%          K.P.Hilgersom@tudelft.nl (correspondence)
% 
%---------------------------------------------------------------------------  

    %% check the existance of a project for this image
    newproject  =   exist([folder '\base grid variables.mat'],'file');
    
    %% check folder structure
    
    % check and, in case it does not exist, create the folder structure needed to save the data  
    if exist([folder '\1 pics'],'dir')==0
        mkdir(folder, '1 pics');
        files=dir(folder);
        if length(files)>2
            movefile([folder '\*'],[folder '\1 pics']);
        end
    end
    if exist([folder '\2 undistorted'],'dir')==0
        mkdir(folder, '2 undistorted');
    end
    if exist([folder '\3 result'],'dir')==0
        mkdir(folder, '3 result');
    end
    if exist([folder '\4 results txt'],'dir')==0
        mkdir(folder, '4 results txt');
    end

    %% set up project (case 0) or load saved project parameters (case 2)

    switch newproject
        case 0
            
            % prompt for input parameters
            base.vr = input('enter rising velocity in m/s: '); % the rising velocity as calibrated for the specific situation
            base.Ninp=input('enter the amount of nozzles used? '); % number of nozzles
            
            % set up base grid and prompt for relative distances of
            %  reference points
            figure(1);clf
            set(gcf,'Position',scrsize);
            set(gca,'Position',[0.1 0.1 0.8 0.8]);
            base.X=[0; 4.2; -0.1; 4.7];base.Y=[0; 0; 4; 4.5];
            base.lvar=  ['a';'b';'c';'d';'e'];base.tadj=...
                        [100 0;0 150;0 150;150 50;-200 250];
            base.pnts=[1 3; 1 2; 3 4; 1 4; 3 2];
            [base] = RABT_CrTFGrid(base);
            set(gca,'Visible','off');
            set(gcf,'PaperPositionMode','auto')
            axis equal
            hold on
            base.txt=annotation('textbox',[0.1 0.96 0.01 0.01],'String',[
                'Please fill in values for the focal length, rising '...
                'velocity, a, b, c, d and e (m.) in the command window']...
                ,'FitBoxToText','on');
            disp('Please fill in values for a, b, c, d and e (m.):');
            base.a = input('a? ');
            base.b = input('b? ');
            base.c = input('c? ');
            base.d = input('d? ');
            base.e = input('e? ');        
            set(base.txt,'Visible','off');
            [base] = RABT_SetTFGrid(base);
            base.da=axis;        
            set(gca,'Units','pixel'); base.dp=get(gca,'Position');
            set(gcf,'Units','pixel'); base.dq=get(gcf,'Position');
            base.fact=(base.da(4)-base.da(3))/base.dp(4);
            if base.a*base.b > 40
                base.scale=10;
            elseif base.a*base.b < 0.1
                base.scale=0.1;
            else
                base.scale=1;
            end
            
            % save base grid and project parameters
            print('-f1', '-dtiff', '-r96', [folder '\base grid']);
            save([folder '\base grid variables'],'base');
            hold off
            
        case 2 % load project parameters
            load([folder '\base grid variables.mat']);
            disp('The former base grid and project constants are used');
    end
end

function [L] = RABT_CrTFGrid(L) % create base grid for warping the image
L.l=ones(5,1);L.t=L.l;
Xd=1000*L.X;Yd=1000*L.Y;
for n=1:5
    L.l(n)=line([Xd(L.pnts(n,1)) Xd(L.pnts(n,2))],[Yd(L.pnts(n,1)) Yd(L.pnts(n,2))],'Marker','o','LineWidth',2);
    L.t(n)=text((Xd(L.pnts(n,1))+Xd(L.pnts(n,2)))/2+L.tadj(n,1),(Yd(L.pnts(n,1))+Yd(L.pnts(n,2)))/2+L.tadj(n,2),L.lvar(n));
end
annotation('arrow',[0.1 0.1],[0.3 0.5]);
annotation('textbox',[0.038 0.28 0.01 0.01],'String','Approximate flow direction','FitBoxToText','on');
end

function [L] = RABT_SetTFGrid(L) % set base grid for warping the image based on input of relative distances of reference points
L.X(2)=L.X(1)+L.b;
[newlat,newlong] = circcirc(L.X(1),L.Y(1),L.a,L.X(2),L.Y(2),L.e);
L.X(3)=newlat(newlong>0);L.Y(3)=newlong(newlong>0);
[newlat,newlong] = circcirc(L.X(1),L.Y(1),L.d,L.X(3),L.Y(3),L.c);
L.X(4)=newlat(newlat>0);L.Y(4)=newlong(newlat>0);
L.X=L.X-min(L.X);L.Y=L.Y-min(L.Y);
Xd=1000*L.X;Yd=1000*L.Y;
for n=1:5
    set(L.l(n),'X',[Xd(L.pnts(n,1)) Xd(L.pnts(n,2))],'Y',[Yd(L.pnts(n,1)) Yd(L.pnts(n,2))]);
    set(L.t(n),'Position',[(Xd(L.pnts(n,1))+Xd(L.pnts(n,2)))/2+L.tadj(n,1)*L.a/5,(Yd(L.pnts(n,1))+Yd(L.pnts(n,2)))/2+L.tadj(n,2)*L.a/5]);
end
end